import json
import math
from PyQt5 import QtCore
from PyQt5.QtCore import pyqtSignal
from PyQt5.QtWidgets import *
from krita import *


class QPushButtonPlus (QPushButton):
    rightClicked = pyqtSignal()
    
    def mouseReleaseEvent(self, evt):
        if evt.button() == QtCore.Qt.RightButton:
            self.rightClicked.emit()
        else:
            super(QPushButtonPlus, self).mouseReleaseEvent(evt)


class QuickBrushSize (DockWidget):
    
    def __init__ (self):
        super().__init__()
        self.setWindowTitle("Quick Brush Size")
        
        self.main_widget = QWidget(self)
        self.setWidget(self.main_widget)
        
        self.main_widget.setStyleSheet("QPushButton { min-width: 58px; max-width: 300px; min-height: 38px; max-height: 200px; background-color: #222222; border: 1px solid #444444; color: #ffffff; } QPushButton:hover:!pressed { background-color: #444444; } QPushButton:pressed { background-color: #6699cc; }")
        
        self.main_layout = QtWidgets.QGridLayout(self.main_widget)
        self.main_layout.setSpacing(0)
        self.main_layout.setContentsMargins(0, 0, 0, 0)
        
        slash_pos = __file__.rfind("/")
        if slash_pos == -1:
            slash_pos = __file__.rfind("\\")
        
        with open(__file__[0:slash_pos] + "/config.json", "r") as json_f:
            self.config = json.load(json_f)
        
        self.circle_icon = QIcon(__file__[0:slash_pos] + "/circle.png")
        
        self.buttons = []
        
        self.bottom_layout = QtWidgets.QHBoxLayout()
        self.bottom_layout.addStretch(1)
        
        self.update_layout(0)
        
        bottom_button_style = "QPushButton { min-width: 18px; max-width: 18px; min-height: 18px; max-height: 18px; background-color: #333333; border: 1px solid #555555; color: #eeeeee; }"
        
        subtract_col_button = QPushButton("▼", self.main_widget)
        subtract_col_button.clicked.connect(self.subtract_col)
        subtract_col_button.setStyleSheet(bottom_button_style)
        self.bottom_layout.addWidget(subtract_col_button)
        
        self.col_label = QLabel(self.main_widget)
        self.col_label.setText(str(self.config["column_count"]))
        self.bottom_layout.addWidget(self.col_label)
        
        add_col_button = QPushButton("▲", self.main_widget)
        add_col_button.clicked.connect(self.add_col)
        add_col_button.setStyleSheet(bottom_button_style)
        self.bottom_layout.addWidget(add_col_button)
        
        add_size_button = QPushButton("+", self.main_widget)
        add_size_button.clicked.connect(self.add_size)
        add_size_button.setStyleSheet(bottom_button_style)
        self.bottom_layout.addWidget(add_size_button)
    
    
    def canvasChanged (self, canvas):
        pass
    
    
    def update_layout (self, idx):
        if len(self.buttons) >= 1:
            for cnt in reversed(range(idx, len(self.buttons))):
                self.buttons[cnt].deleteLater()
                self.buttons.pop(cnt)
            
            self.main_layout.removeItem(self.bottom_layout)
        
        for cnt in range(idx, len(self.config["brush_sizes"])):
            self.buttons.append(QPushButtonPlus(str(self.config["brush_sizes"][cnt]), self.main_widget))
            self.buttons[cnt].clicked.connect(self.change_size)
            self.buttons[cnt].rightClicked.connect(self.delete_size)
            self.buttons[cnt].setFlat(True)
            
            if self.config["brush_sizes"][cnt] > 24:
                icon_size = 24
            else:
                icon_size = self.config["brush_sizes"][cnt]
            
            self.buttons[cnt].setIcon(self.circle_icon)
            self.buttons[cnt].setIconSize(QtCore.QSize(icon_size, icon_size))
            
            self.main_layout.addWidget(self.buttons[cnt], math.floor(cnt / self.config["column_count"]), cnt % self.config["column_count"])
        
        self.main_layout.addLayout(self.bottom_layout, math.ceil(len(self.config["brush_sizes"]) / self.config["column_count"]), 0, 1, self.config["column_count"])
    
    
    def change_size (self):
        size_val = int(self.sender().text())
        current_view = Krita.instance().activeWindow().activeView()
        current_view.setBrushSize(size_val)
    
    
    def save_config (self):
        slash_pos = __file__.rfind("/")
        if slash_pos == -1:
            slash_pos = __file__.rfind("\\")
        
        json_f = open(__file__[0:slash_pos] + "/config.json", "w", encoding="utf-8")
        json.dump(self.config, json_f, ensure_ascii=False, indent=4)
    
    
    def subtract_col (self):
        if self.config["column_count"] >= 2:
            self.config["column_count"] -= 1
            self.col_label.setText(str(self.config["column_count"]))
            
            self.update_layout(0)
            
            self.save_config()
    
    
    def add_col (self):
        self.config["column_count"] += 1
        self.col_label.setText(str(self.config["column_count"]))
        
        self.update_layout(0)
        
        self.save_config()
    
    
    def add_size (self):
        current_view = Krita.instance().activeWindow().activeView()
        size_val = math.floor(current_view.brushSize())
        
        if size_val < 1:
            size_val = 1
        
        if QMessageBox.question(None, "ブラシサイズの追加", "現在のブラシサイズ " + str(size_val) + "px をリストに追加しますか？", QMessageBox.Yes, QMessageBox.Cancel) == QMessageBox.Yes:
            if size_val in self.config["brush_sizes"]:
                QMessageBox.warning(None, "Error", "既に存在するブラシサイズです", QMessageBox.Yes)
            else:
                idx = len(self.config["brush_sizes"])
                for cnt in range(idx):
                    if self.config["brush_sizes"][cnt] > size_val:
                        idx = cnt
                        break
                
                self.config["brush_sizes"].insert(idx, size_val)
                self.update_layout(idx)
                
                self.save_config()
    
    
    def delete_size (self):
        size_val = int(self.sender().text())
        
        if len(self.config["brush_sizes"]) == 1:
            QMessageBox.warning(None, "Error", "ブラシサイズを全て削除することはできません。", QMessageBox.Yes)
        elif QMessageBox.question(None, "ブラシサイズの削除", "ブラシサイズ " + str(size_val) + "px を削除しますか？", QMessageBox.Yes, QMessageBox.Cancel) == QMessageBox.Yes:
            idx = self.config["brush_sizes"].index(size_val)
            self.config["brush_sizes"].pop(idx)
            self.update_layout(idx)
            
            self.save_config()


Krita.instance().addDockWidgetFactory(DockWidgetFactory("Quick Brush Size", DockWidgetFactoryBase.DockRight, QuickBrushSize))
